"use client";

import { useSelector } from "react-redux";
import { RootState } from "../../../store";
import { formatDate } from "../../../common/DateFormant";
import { useState, Fragment, useEffect } from "react";
import {
  addTickets,
  addReview,
  trackOrder,
  cancelOrder,
} from "../../../api/api";
import { useDispatch } from "react-redux";
import { unwrapResult } from "@reduxjs/toolkit";
import { FaStar } from "react-icons/fa";
import { toast } from "react-toastify";
import { Menu, Transition } from "@headlessui/react";
import { ChevronDownIcon } from "@heroicons/react/20/solid";
import { useParams, useRouter } from "next/navigation";
import { Formik, Form, Field, ErrorMessage } from "formik";
import * as Yup from "yup";
import Link from "next/link";
import Image from "next/image";
import CancelOrderPopupp from "../../../common/CancelOrderPopup";

function OrderStatus() {
  const router = useRouter();
  const params = useParams();
  const productId = params?.id as string;
  const { orderData } = useSelector(
    (state: RootState) => state.OrderTrackReducer
  );

  const ordersDetail = orderData?.data?.orders;
  const [showTicketForm, setShowTicketForm] = useState(false);
  const dispatch = useDispatch<any>();
  const [errorMessage, setErrorMessage] = useState("");
  const [open, setOpen] = useState(false);
  const [rating, setRating] = useState(0);
  const [comment, setComment] = useState("");
  const [ratingError, setRatingError] = useState("");
  const [commentError, setCommentError] = useState("");
  const [isPopupOpen, setIsPopupOpen] = useState(false);

  const handleOpen = () => setOpen(true);
  const handleClose = () => {
    setOpen(false);
    setErrorMessage("");
  };

  const steps = [
    { label: "Bidding Process", status: "completed" },
    { label: "Accepted", status: "pending" },
    { label: "In Progress", status: "pending" },
    { label: "In Transit", status: "pending" },
    { label: "Completed", status: "pending" },
  ];

  useEffect(() => {
    if (productId) {
      dispatch(trackOrder(productId));
    }
  }, [dispatch, productId]);

  useEffect(() => {
    const routesToPrefetch = ["/profile/tickets"];

    routesToPrefetch.forEach((route) => {
      router.prefetch(route);
    });
  }, [router]);

  const updateStepStatus = (orderStatus: string) => {
    let isCompleted = true;

    return steps.map((step) => {
      if (isCompleted) {
        step.status = "completed";
      }

      if (step.label === orderStatus) {
        step.status = "current";
        isCompleted = false;
      }

      return step;
    });
  };
  const orderStatusData = ordersDetail?.order_status;
  const updatedSteps = updateStepStatus(orderStatusData);

  const ticketValidationSchema = Yup.object({
    priority: Yup.string().required("Priority is required"),
    subject: Yup.string().required("Subject is required"),
    description: Yup.string().required("Description is required"),
  });

  const handleAddTicket = async (values: any) => {
    if (!values.subject.trim() || !values.description.trim()) {
      alert("Please fill in both subject and description.");
      return;
    }

    const listingId = ordersDetail?.listing_id;
    if (!listingId) {
      alert("Listing ID is missing. Cannot create ticket.");
      return;
    }

    try {
      const payload: any = {
        listing_id: listingId,
        category: "dispute",
        subject: values.subject,
        message: values.description,
        priority: values.priority,
      };

      const resultAction = await dispatch(addTickets(payload));

      const response = unwrapResult(resultAction);

      if (response.success) {
        router.push("/profile/tickets");
      } else {
        const errorMessage = response.message || "Unknown error occurred.";
        alert(`Failed to create ticket: ${errorMessage}`);
      }
    } catch {
      alert("An error occurred while creating the ticket. Please try again.");
    }
  };

  const handleSubmit = async (e: React.FormEvent<HTMLFormElement>) => {
    setErrorMessage("");
    e.preventDefault();
    // Reset errors
    setRatingError("");
    setCommentError("");

    let hasError = false;

    if (rating === 0) {
      setRatingError("Rating is required.");
      hasError = true;
    }

    if (!comment.trim()) {
      setCommentError("Comment is required.");
      hasError = true;
    }

    if (hasError) return;

    try {
      const listing_id = ordersDetail?.listing_id;
      const reviewData = { listing_id, rating, comment };

      const response = await dispatch(addReview(reviewData) as any);
      console.log("check response", response);

      if (response.type.endsWith("fulfilled")) {
        toast("Review submitted successfully");

        setRating(0);
        setComment("");

        handleClose();
      } else if (response?.payload) {
        toast(response?.payload);
        setRating(0);
        setComment("");
        setErrorMessage(response?.payload);
      } else {
        toast("Something went wrong while submitting the review.");
        setErrorMessage("Something went wrong while submitting the review.");
      }
    } catch {
      alert("Unexpected error occurred.");
      setErrorMessage("Unexpected error occurred.");
    }
  };

  const handleRatingClick = (star: number) => {
    setRating(star);
    if (ratingError) setRatingError(""); // Clear rating error immediately
  };

  const handleCommentChange = (e: React.ChangeEvent<HTMLTextAreaElement>) => {
    setComment(e.target.value);
    if (commentError && e.target.value.trim()) {
      setCommentError(""); // Clear comment error if not empty
    }
  };

  const handleCancelOrder = async ({
    orderId,
    reason,
  }: {
    orderId: string | number;
    reason: string;
  }) => {
    try {
      const response = await dispatch(
        cancelOrder({ orderId: orderId, cancellation_reason: reason }) as any
      );

      if (response.type.endsWith("fulfilled")) {
        toast.success("Order cancelled successfully");
        setIsPopupOpen(false); // ✅ close popup after success
        // 👉 Optionally refresh order tracking
        dispatch(trackOrder(productId));
      } else {
        toast.error(response?.payload || "Failed to cancel order");
      }
    } catch (error) {
      console.error("Cancel order error:", error);

      toast.error("Unexpected error occurred while cancelling order");
    }
  };

  return (
    <>
      <div className="max-w-7xl w-full m-auto min-h-screen bg-gray-50 pt-4 flex flex-col items-center">
        <div className=" bg-white w-full rounded-lg">
          <div className="mb-4 flex justify-between xs:p-3">
            <h1 className="xs:text-sm font-semibold text-black lg:text-4xl flex items-center gap-4 cursor-pointer">
              {" "}
              <Link href="/profile/listing">
                <svg
                  xmlns="http://www.w3.org/2000/svg"
                  fill="none"
                  viewBox="0 0 24 24"
                  strokeWidth="1.5"
                  stroke="currentColor"
                  className="lg:size-9 xs:size-5"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    d="M10.5 19.5 3 12m0 0 7.5-7.5M3 12h18"
                  />
                </svg>{" "}
              </Link>
              Bid Details
            </h1>
            <button
              className="animation-custom-button xs:text-sm"
              onClick={() => setShowTicketForm(!showTicketForm)}
            >
              {showTicketForm ? "Create Ticket" : "Create Ticket"}
            </button>

            {showTicketForm && (
              <div className="fixed inset-0 flex items-center justify-center bg-black bg-opacity-50 z-50 p-3">
                <div className="bg-white p-6 rounded-lg shadow-lg w-full max-w-lg relative">
                  <button
                    className="absolute top-0 right-3 text-black font-normal cursor-pointer hover:text-gray-800 text-4xl p-0"
                    onClick={() => setShowTicketForm(false)}
                  >
                    &times;
                  </button>

                  <h2 className="text-xl font-semibold mb-4 text-center">
                    Create New Ticket
                  </h2>

                  <Formik
                    initialValues={{
                      priority: "",
                      subject: "",
                      description: "",
                    }}
                    validationSchema={ticketValidationSchema}
                    onSubmit={(values, { resetForm }) => {
                      handleAddTicket(values);
                      resetForm();
                      setShowTicketForm(false);
                    }}
                  >
                    {({ setFieldValue, values, touched, errors }) => (
                      <Form>
                        <div className="mb-4">
                          <Menu
                            as="div"
                            className="relative inline-block w-full text-left"
                          >
                            <div>
                              <Menu.Button className="inline-flex justify-between items-center w-full rounded-md bg-white px-4 py-2 border border-gray-300 text-sm text-gray-700 shadow-sm hover:bg-gray-50">
                                {values.priority || "Select Priority"}
                                <ChevronDownIcon className="ml-2 h-5 w-5 text-gray-500" />
                              </Menu.Button>
                            </div>
                            <Transition
                              as={Fragment}
                              enter="transition ease-out duration-100"
                              enterFrom="transform opacity-0 scale-95"
                              enterTo="transform opacity-100 scale-100"
                              leave="transition ease-in duration-75"
                              leaveFrom="transform opacity-100 scale-100"
                              leaveTo="transform opacity-0 scale-95"
                            >
                              <Menu.Items className="absolute z-10 mt-2 w-full rounded-md bg-white shadow-lg ring-1 ring-black ring-opacity-5 focus:outline-none overflow-hidden">
                                <div className="">
                                  {["High", "Low"].map((level) => (
                                    <Menu.Item key={level}>
                                      {({ active }) => (
                                        <button
                                          type="button"
                                          onClick={() =>
                                            setFieldValue("priority", level)
                                          }
                                          className={`${
                                            active
                                              ? "bg-navy text-white dark:bg-gray-600"
                                              : "text-gray-900 dark:text-white"
                                          }  block w-full text-left cursor-pointer px-4 py-3 text-sm text-black rounded-none border-b border-gray-200`}
                                        >
                                          {level}
                                        </button>
                                      )}
                                    </Menu.Item>
                                  ))}
                                </div>
                              </Menu.Items>
                            </Transition>
                          </Menu>
                          {touched.priority && errors.priority && (
                            <div className="text-red-500 text-sm mt-1">
                              {errors.priority}
                            </div>
                          )}
                        </div>

                        <div className="mb-3">
                          <Field
                            type="text"
                            name="subject"
                            placeholder="Subject"
                            className="w-full p-3 border border-gray-300 rounded-md focus:outline-none focus:ring-2"
                          />
                          <ErrorMessage
                            name="subject"
                            component="div"
                            className="text-red-500 text-sm mt-1"
                          />
                        </div>

                        <div className="mb-3">
                          <Field
                            as="textarea"
                            name="description"
                            placeholder="Enter Ticket Description"
                            className="w-full p-3 border border-gray-300 rounded-md focus:outline-none focus:ring-2"
                          />
                          <ErrorMessage
                            name="description"
                            component="div"
                            className="text-red-500 text-sm mt-1"
                          />
                        </div>

                        <button
                          type="submit"
                          className="bg-navy text-white px-4 py-2 rounded-md w-full"
                        >
                          Submit Ticket
                        </button>
                      </Form>
                    )}
                  </Formik>
                </div>
              </div>
            )}
          </div>
          <hr />
          <section className="relative md:px-0 xs:p-3">
            <div className="flex w-full md:max-w-full md:flex lg:max-w-full lg:flex md:mb-4 lmd:mt-3 rounded-lg bg-white sm:m-auto sm:mb-4 sm:max-w-[28rem] lg:p-3 lg:gap-3 md:gap-5 cursor-pointer justify-between">
              <div className=" bg-white lg:w-8/12 rounded-b lg:rounded-b-none lg:rounded-r p-1 flex justify-between leading-normal ">
                <div>
                  <div className="xs:text-lg text-gray-900 font-bold lg:text-3xl mb-1 text-start lg:pt-1 capitalize">
                    {ordersDetail?.listing?.product_name}
                  </div>

                  <div className=" mb-3 lg:mb-1 text-gray-600 lg:pt-1">
                    <p className="flex items-center gap-1 xs:text-sm lg:text-2xl font-medium">
                      <span>{formatDate(ordersDetail?.created_at)} </span> |
                      <span>{ordersDetail?.listing?.location}</span>
                    </p>
                  </div>
                  <p className="hidden text-black-700 text-start lg:text-lg font-bold text-[#373737] lg:block lg:pt-1">
                    {ordersDetail?.listing?.description}
                  </p>
                  <p className="text-green-500 md:text-2xl font-semibold text-start xs:text-lg mt-2 flex flex-col">
                    <span>
                      {ordersDetail?.currency === "USD" ? "US$" : "C$"}{" "}
                      {ordersDetail?.amount}
                    </span>
                    {ordersDetail?.listing?.currency !=
                      ordersDetail?.currency && (
                      <span className="text-gray-500 font-semibold text-base">
                        {ordersDetail?.listing?.currency === "USD"
                          ? "US$"
                          : "C$"}{" "}
                        {ordersDetail?.bid_amount}
                      </span>
                    )}
                  </p>
                </div>
              </div>

              <div className="relative lg:w-4/12 lg:h-[212px] xs:h-[114px] xs:w-[122px] lg:p-0 p-2 rounded-lg overflow-hidden">
                <Image
                  unoptimized
                  src={ordersDetail?.listing?.images[0]}
                  alt={ordersDetail?.listing?.product_name || "Listing image"}
                  fill
                  className="object-cover rounded-lg"
                  sizes="(max-width: 768px) 122px, 248px"
                />
              </div>
            </div>
          </section>
          <hr />
          {ordersDetail?.order_status == "Cancelled" ? (
            // ✅ Show only cancelled details
            <div className="p-6 mb-6">
              <h3 className="text-xl font-bold text-red-600 mb-2">
                Order Cancelled
              </h3>
              <p className="text-gray-700 capitalize">
                <span className="font-semibold">Status:</span>{" "}
                {ordersDetail?.order_status}
              </p>
              <p className="text-gray-700 capitalize">
                <span className="font-semibold">Reason:</span>{" "}
                {ordersDetail?.cancellation_reason || "N/A"}
              </p>
              <p className="text-gray-700 capitalize">
                <span className="font-semibold">Cancelled By:</span>{" "}
                {ordersDetail?.cancelled_by || "N/A"}
              </p>
              <p className="text-gray-700">
                <span className="font-semibold">Cancelled On:</span>{" "}
                {ordersDetail?.cancelled_at
                  ? formatDate(ordersDetail?.cancelled_at)
                  : "N/A"}
              </p>
            </div>
          ) : (
            // ✅ Show steps + Cancel button if not cancelled
            <>
              <div className="hidden lg:block p-6">
                <h3 className="text-3xl text-start text-black font-semibold py-8 mb-2">
                  Order Tracking
                </h3>
                <div className="relative px-4 py-6 mx-10">
                  <div className="absolute top-[60%] left-16 right-16 h-1 bg-green-500 -translate-y-1/2 z-0"></div>
                  <div className="absolute top-1/2 left-16 h-1 bg-green-500 -translate-y-1/2 z-0 transition-all duration-300"></div>

                  <div className="flex justify-between relative z-10 ">
                    {steps.map((step, index) => {
                      const isCompleted = step.status === "completed";
                      const isCurrent = step.status === "current";
                      const isPending = step.status === "pending";

                      return (
                        <div
                          key={index}
                          className="flex flex-col-reverse items-center gap-3"
                        >
                          <div
                            className={`
                    w-10 h-10 rounded-full flex items-center justify-center mb-2
                    ${
                      isCompleted || isCurrent
                        ? "bg-green-900"
                        : "border-2 border-gray-300 bg-white"
                    }
                    ${isCurrent ? "ring-2 ring-green-300" : ""}
                  `}
                          >
                            {(isCompleted || isCurrent) && (
                              <svg
                                className="w-5 h-5 text-white"
                                fill="none"
                                viewBox="0 0 24 24"
                                stroke="currentColor"
                              >
                                <path
                                  strokeLinecap="round"
                                  strokeLinejoin="round"
                                  strokeWidth={2}
                                  d="M5 13l4 4L19 7"
                                />
                              </svg>
                            )}
                          </div>
                          <span
                            className={`
                    text-base font-medium text-center
                    ${isCompleted ? "text-gray-800" : ""}
                    ${isCurrent ? "text-green-600 font-bold" : ""}
                    ${isPending ? "text-gray-400" : ""}
                  `}
                          >
                            {step.label}
                          </span>
                        </div>
                      );
                    })}
                  </div>
                </div>
              </div>

              {/* Mobile view */}
              <div className="block lg:hidden p-6">
                <h3 className="lg:text-3xl text-start text-black font-semibold py-8 mb-2-">
                  Order Tracking
                </h3>
                <div className="relative">
                  <div className="absolute left-7 z-10 top-0 h-full w-0.5 bg-green-500"></div>
                  {updatedSteps.map((step, index) => (
                    <div
                      key={index}
                      className={`flex items-start mb-10 relative rounded-md pl-3 ${
                        step.status === "current"
                          ? "bg-[#ecf5ec] p-3 rounded-lg"
                          : ""
                      }`}
                    >
                      <div
                        className={`flex items-center justify-center w-8 h-8 rounded-full z-10 ${
                          step.status === "completed" ||
                          step.status === "current"
                            ? "bg-[#0B7D0C] text-white"
                            : "border-2 border-gray-300 bg-white"
                        }`}
                      >
                        {step.status === "completed" ||
                        step.status === "current" ? (
                          <svg
                            className="h-6 text-white"
                            fill="none"
                            stroke="currentColor"
                            strokeWidth="2"
                            viewBox="0 0 24 24"
                          >
                            <path
                              strokeLinecap="round"
                              strokeLinejoin="round"
                              d="M5 13l4 4L19 7"
                            ></path>
                          </svg>
                        ) : (
                          <span className="text-gray-500"></span>
                        )}
                      </div>
                      <p
                        className={`text-xl font-medium ml-3 ${
                          step.status === "completed"
                            ? "text-gray-900"
                            : step.status === "current"
                            ? "text-black font-bold"
                            : "text-gray-500"
                        }`}
                      >
                        {step.label}
                      </p>
                    </div>
                  ))}
                </div>
              </div>

              {/* Cancel Order Button */}
              <div className="flex justify-start p-6 mb-6">
                <button
                  onClick={() => setIsPopupOpen(true)}
                  className="bg-red-500 text-white px-4 py-2 rounded-md"
                >
                  Cancel Order
                </button>

                <CancelOrderPopupp
                  isOpen={isPopupOpen}
                  onClose={() => setIsPopupOpen(false)}
                  orderId={ordersDetail?.id}
                  onConfirm={handleCancelOrder}
                />
              </div>
            </>
          )}

          <hr />
          <div className="py-3 px-4">
            <h3 className="xs:text-lg font-medium mb-4 lg:text-2xl text-[#A2A2A1] text-start">
              Seller Details
            </h3>

            <div className="flex justify-between items-center">
              <div>
                <p className="text-[#373737] xs:text-lg lg:text-xl text-start font-semibold">
                  Name:
                  <span className="text-[#373737] text-base capitalize">
                    {" "}
                    {ordersDetail?.seller?.name}{" "}
                  </span>
                </p>
                <p className="text-[#2E2E2E] text-start lg:text-xl font-semibold">
                  Email:{" "}
                  <span className="text-[#373737] text-base ">
                    {" "}
                    {ordersDetail?.seller?.email}{" "}
                  </span>
                </p>
                <p className="text-[#2E2E2E] text-start lg:text-xl font-semibold capitalize">
                  Phone No:{" "}
                  <span className="text-[#373737] text-base capitalize">
                    {" "}
                    {ordersDetail?.seller?.phone}{" "}
                  </span>
                </p>
                <p className="text-[#2E2E2E] text-start lg:text-xl font-semibold capitalize">
                  Location:{" "}
                  <span className="text-[#373737] text-base capitalize">
                    {[
                      ordersDetail?.seller?.default_address?.house_number,
                      ordersDetail?.seller?.default_address?.city,
                      ordersDetail?.seller?.default_address?.postal_code,
                      ordersDetail?.seller?.default_address?.country === "USA"
                        ? ordersDetail?.seller?.default_address?.state
                        : ordersDetail?.seller?.default_address?.province,
                      ordersDetail?.seller?.default_address?.country,
                    ]
                      .filter((val) => val && val.trim() !== "")
                      .join(", ")}
                  </span>
                </p>

                <div className="mt-4 text-start text-[#373737] text-lg">
                  <p className="font-semibold text-[#A2A2A1] text-xl lg:text-2xl">
                    Next Steps for Transfer
                  </p>
                  <ul className="list-decimal list-inside space-y-4 mt-2">
                    <li>
                      <span className="font-semibold text-[#373737] text-lg">
                        Confirm Details and Coordinate Payment
                      </span>
                      <p className="text-[#373737] text-base mt-1">
                        Connect with the seller to confirm all purchase details
                        and agree on a secure payment method.
                      </p>
                      <ul className="list-disc list-inside ml-6 mt-1 space-y-1">
                        <li className="text-[#373737] text-base">
                          Before making any payment, verify the seller's
                          identity and ensure the item matches the description
                          provided.
                        </li>
                        <li className="text-[#373737] text-base">
                          Consider exchanging user IDs or contact details to
                          ensure trust and transparency.
                        </li>
                        <li className="text-[#373737] text-base">
                          Use trusted payment platforms that offer buyer
                          protection whenever possible.
                        </li>
                      </ul>
                    </li>

                    <li>
                      <span className="font-semibold text-[#373737] text-lg">
                        Arrange Pickup or Delivery
                      </span>
                      <p className="text-[#373737] text-base mt-1">
                        Coordinate the pickup or delivery of the purchased item.
                        Our support team is available if you need assistance.
                      </p>
                    </li>

                    <li>
                      <span className="font-semibold text-[#373737] text-lg">
                        Complete Documentation
                      </span>
                      <p className="text-[#373737] text-base mt-1">
                        Fill out any required paperwork to finalize the
                        transfer. For further help, you can always submit a
                        support ticket.
                      </p>
                    </li>
                  </ul>
                </div>
              </div>
            </div>
          </div>

          <hr />
          <div className="mb-6 mt-7  xs:p-3">
            <h3 className="xs:text-lg font-medium mb-4 lg:text-2xl text-[#A2A2A1] text-start">
              Shipping Details
            </h3>
            <div className="flex justify-between items-center">
              <p>
                <br />
                <p className="text-[#2E2E2E] text-start lg:text-2xl font-medium capitalize">
                  {" "}
                  #{ordersDetail?.address?.house_number}{" "}
                  {ordersDetail?.address?.locality}{" "}
                </p>
              </p>
            </div>
          </div>
          <div className="mb-4  xs:p-3">
            <div className="flex justify-between font-medium">
              <p className="text-[#373737] lg:text-2xl font-medium">
                Order Total
              </p>
              <p className="text-[#373737] lg:text-2xl font-medium flex flex-col">
                <span>
                  {ordersDetail?.currency === "USD" ? "US$" : "C$"}{" "}
                  {ordersDetail?.converted_price}
                </span>
                {ordersDetail?.listing?.currency != ordersDetail?.currency && (
                  <span className="text-gray-500 font-semibold text-base">
                    {ordersDetail?.listing?.currency === "USD" ? "US$" : "C$"}{" "}
                    {ordersDetail?.bid_amount}
                  </span>
                )}
              </p>
            </div>
          </div>
          <hr />
          <div className="flex flex-col py-6">
            <div className="flex justify-between xs:p-3">
              <p className="text-[#373737] lg:text-xl xs:text-lg font-medium text-start py-4">
                Review
              </p>
              <button onClick={handleOpen} className="animation-custom-button">
                Add Review
              </button>
            </div>
          </div>

          {open && (
            <div className="fixed inset-0 flex items-center justify-center bg-black bg-opacity-50 z-20">
              <div className="bg-white rounded-lg p-8 xs:w-[300px] lg:w-[340px]">
                <div className="">
                  {errorMessage && (
                    <div className="text-sm text-red-600 text-center font-semibold pb-2">
                      {errorMessage}
                    </div>
                  )}
                  <h3 className="text-black font-bold text-center mb-2">
                    Review
                  </h3>
                  <form onSubmit={handleSubmit}>
                    <div className="mb-4 text-center">
                      <label className="block text-gray-700 mb-1">
                        Rating:
                      </label>
                      <div className="flex justify-center">
                        {[1, 2, 3, 4, 5].map((star) => (
                          <FaStar
                            key={star}
                            className={`cursor-pointer text-2xl ${
                              rating >= star
                                ? "text-yellow-400"
                                : "text-gray-300"
                            }`}
                            // onClick={() => setRating(star)}
                            onClick={() => handleRatingClick(star)}
                          />
                        ))}
                      </div>
                      {ratingError && (
                        <div className="text-red-600 font-medium text-sm mt-1 text-center">
                          {ratingError}
                        </div>
                      )}
                    </div>
                    <div className="mb-4">
                      <label className="block text-gray-700">Comment:</label>
                      <textarea
                        value={comment}
                        // onChange={(e) => setComment(e.target.value)}
                        onChange={handleCommentChange}
                        className="mt-1 p-2 w-full border rounded text-black"
                      ></textarea>
                      {commentError && (
                        <div className="text-red-600 font-medium text-sm mt-1 text-center">
                          {commentError}
                        </div>
                      )}
                    </div>
                    <div className="flex justify-between">
                      <button
                        type="submit"
                        className="bg-blue-900 hover:bg-blue-800 text-white font-bold py-2 px-4 rounded"
                      >
                        Submit Review
                      </button>
                      <button
                        type="button"
                        onClick={handleClose}
                        className="bg-red-500 text-white font-bold py-2 px-4 rounded"
                      >
                        Close
                      </button>
                    </div>
                  </form>
                </div>
              </div>
            </div>
          )}
        </div>
      </div>
    </>
  );
}

export default OrderStatus;
